/*
 * ACE/vis - Visualization of Flow and Transport
 *
 * Paul J. Turner and Antonio M. Baptista
 *
 * Copyright 1990-2003 Oregon Health and Science University
 * All Rights Reserved
 *
 */

/*
 *
 * defines and typedefs for xmvis
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <time.h>
#include <string.h>
#include "elio.h"

double hypot(double x, double y);

/*
 * include defines generated by yacc
 */
#ifndef GRAMMAR
#include "gram.h"
#endif

#ifndef lint
static char RCSid_defines[] = "$Id: defines.h,v 1.28 2007/01/13 21:48:46 pturner Exp $";
#endif

/*
 * some constants
 *
 */
#define MAXGRAPH 10		/* max number of graphs */
#define MAXPLOT 30		/* max number of XY data sets */
#define MAXFIT 15		/* max number degree of polynomial fit */
#define MAX_ZOOM_STACK 20	/* max stack depth for world stack */
#define MAXREGION 5		/* max number of regions */
#define MAX_TICK_LABELS 35	/* max number of user defined ticks/labels */
#define MAX_STRING_LENGTH 256	/* max length for strings */
#define MAX_LINESTYLE 5		/* max number of linestyles */
#define MAX_LINEWIDTH 10	/* max number of line widths */
#define MAX_SET_COLS 6		/* max number of columns per XY set */
#define MAXSCRATCH 500		/* max number of elements in scatch arrays */

/* data object limits */
#define MAXGRIDS 5		/* max number of grids */
#define MAXGRIDTS 5		/* max number of grids */
#define MAXGRIDBOUNDS 10000        /* max number of bnd segments*/
#define MAXBOUNDS 20000		/* max number of nodes on each boundary segment*/
#define MAXTEANL 5
#define MAXFREQ 65		/* max number of frequencies for TEANL */
#define MAXADCIRC 5		/* max number of ADCIRC flows */
#define MAXADCIRC3D 50		/* max number of ADCIRC3D flows and
				 * elevations */
#define MAXTRANS 5		/* max number of transects */
#define MAXELA 5		/* max number of ELA concentrations */
#define MAXFLOW3D 5		/* max number of 3D flows */
#define MAXPATHLINES 10		/* max number of pathlines */
#define MAXDROGS 10000		/* max number of particles */
#define MAXTRACK 25		/* max number of tracklines */
#define MAXTIDESTATIONS 50	/* max number of tide stations */
#define MAXTIMEHIST 50		/* max number of time histories of elevations
				 * + velocities */
#define MAXVELHIST 5		/* max number of time histories of velocities */
#define MAXSLICES 10		/* max number of slices */
#define MAXFLUX 10		/* max number of flux markers */
#define MAXSTATIONS 20		/* max number of stations */
#define MAXFISHES 500		/* max number of fishes */

/* graph display limits */
#define MAXHISTMARKERS 10	/* max number of time history markers */
#define MAXVELMARKERS 10	/* max number of time history markers */
#define MAXZOOMBOXES 10		/* max number of zoom boxes */
#define MAXELEVMARKERS 20	/* maximum number of elevation markers */
#define MAXSLICEBOXES 10	/* max number of slice boxes */
#define MAXFLUXBOXES 10		/* max number of flux boxes */
#define MAXISOLINES 16		/* max number of isolines */

#define MAXBOXES 500		/* max number of boxes */
#define MAXLINES 500		/* max number of lines */
#define MAXSTR 500		/* max number of strings */
#define MAXSTRLEN 256		/* max string length */
#define MAXSYM 26		/* max number of symbols */

#define TEANL0 0
#define TEANL1 1
#define TEANL2 2
#define TEANL3 3
#define TEANL4 4
#define ADCIRC0 0
#define ADCIRC1 1
#define ADCIRC2 2
#define ADCIRC3 3
#define ADCIRC4 4
#define ELA0 0
#define ELA1 1
#define ELA2 2
#define ELA3 3
#define ELA4 4
#define TIMEHIST0 0
#define TIMEHIST1 1
#define TIMEHIST2 2
#define TIMEHIST3 3
#define TIMEHIST4 4

/* for display buttons on left panel of main */
#define NDISPLAY 7
#define EDIT_GRID 0
#define EDIT_BOUNDARY 1
#define EDIT_GRID_NODE_NUMBERS 2
#define EDIT_GRID_ELEMENT_NUMBERS 3
#define EDIT_GRID_FILLED 4
#define EDIT_GRID_ISOLINES 5
#define EDIT_GRID_DEPTHS 6

#ifndef M_PI
#       define M_PI  3.14159265358979323846
#endif

/* these should go away */
#define BIG 1.0e+307
#define MBIG -1.0e+307

#define HISTOSYM 19		/* plotsymbol for histograms */

/*
 * types of tick displays
 */
#define X_AXIS 0
#define Y_AXIS 1
#define ZX_AXIS 2
#define ZY_AXIS 3
#define XA_AXIS 4
#define YA_AXIS 5
#define POLAR_RADIUS 6
#define POLAR_ANGLE 7

/*
 * grid types
 */
#define GRID_Z 0
#define GRID_UV 1
#define GRID_ZUV 2
#define GRID_ZUVW 3
#define GRID_ZW 4

/* graphics output to the following */
#define GR_X	 	0
#define GR_PS_L 	1	/* PostScript landscape */
#define GR_PS_P 	2	/* PostScript portrait */
#define GR_MIF_L 	3	/* mif landscape */
#define GR_MIF_P 	4	/* mif portrait */
#define GR_HPGL_L 	5	/* hpgl landscape */
#define GR_HPGL_P 	6	/* hpgl portrait */
#define GR_HPGLB_L 	7	/* hpgl 11x17 landscape */
#define GR_HPGLB_P 	8	/* hpgl 11x17 portrait */
#define GR_XWD 		9	/* X window dump */

/* set HDEV to the default hardcopy device */
#ifndef HDEV
#	define HDEV GR_PS_L
#endif

/* TDEV is always X */
#define TDEV GR_X

#define FREE 0
#define LANDSCAPE 1
#define PORTRAIT 2

/*
 * event handling defines
 */
#define COMP_AREA 1
#define COMP_PERIMETER 2
#define COPY_OBJECT 3
#define COPY_OBJECT1ST 4
#define COPY_OBJECT2ND 5
#define CUT_OBJECT 6
#define DEFINE_REGION 7
#define DEFINE_VIEW 8
#define DEFINE_ZOOM 9
#define DELETE_DROGUE 10
#define DEL_CONC 11
#define DEL_ELEV 12
#define DEL_HIST 13
#define DEL_OBJECT 14
#define DEL_SAMPLE_ADCIRC 15
#define DEL_SAMPLE_TEANL 16
#define DEL_VEL 17
#define DEL_ZOOM_BOX 18
#define EDIT_ADCIRC_ELEV 19
#define EDIT_HIST 20
#define EDIT_OBJECT 21
#define EDIT_TEANL_ELEV 23
#define GET_ADCIRC_ELEV 24
#define GET_ADCIRC_FLOW 25
#define GET_ALL 26
#define GET_BACK_DEPTH 27
#define GET_ELA_CONC 28
#define GET_ELEMENT 29
#define GET_GRID_DEPTH 30
#define GET_NEAREST_ELEMENT 31
#define GET_NEAREST_NODE 32
#define GET_TEANL_ELEV 33
#define GET_TEANL_FLOW 34
#define HIDE_OBJECT 35
#define MAKE_BOX 36
#define MAKE_BOX_1ST 37
#define MAKE_BOX_2ND 38
#define MAKE_LINE 39
#define MAKE_LINE_1ST 40
#define MAKE_LINE_2ND 41
#define MOVE_HIST 42
#define MOVE_OBJECT 43
#define MOVE_OBJECT_1ST 44
#define MOVE_OBJECT_2ND 45
#define PASTE_OBJECT 46
#define PICK_BATH 47
#define PICK_DROGUE 48
#define PLACE_ADCIRC_ELEV1ST 49
#define PLACE_ADCIRC_ELEV2ND 50
#define PLACE_BATH_LEGEND 51
#define PLACE_CLOCK 52
#define PLACE_CONC_LEGEND 53
#define PLACE_DROGUE 54
#define PLACE_ELEV_LEGEND 55
#define PLACE_HIST1ST 56
#define PLACE_HIST2ND 57
#define PLACE_INFO 58
#define PLACE_MAPSCALE 59
#define PLACE_TEANL_ELEV1ST 60
#define PLACE_TEANL_ELEV2ND 61
#define PLACE_TEXT 62
#define PLACE_TIMELINE 63
#define PLACE_TIMESTAMP 64
#define PLACE_TITLE 65
#define PLACE_VSCALE 66
#define PLACE_WSCALE 180
#define RUBBER_BAND 67
#define SAMPLE_ADCIRC 68
#define SAMPLE_TEANL 69
#define SELECT_REGION 70
#define SEL_POINT 71
#define SLICE_ADCIRC_ELEV1ST 72
#define SLICE_ADCIRC_ELEV2ND 73
#define SLICE_ADCIRC_FLOW1ST 74
#define SLICE_ADCIRC_FLOW2ND 75
#define SLICE_BATH1ST 76
#define SLICE_BATH2ND 77
#define SLICE_BOX1ST 78
#define SLICE_BOX2ND 79
#define SLICE_BOX3RD 80
#define SLICE_ELA1ST 81
#define SLICE_ELA2ND 82
#define SLICE_LINE1ST 83
#define SLICE_LINE2ND 84
#define SLICE_POLY 85
#define SLICE_TEANL_ELEV1ST 86
#define SLICE_TEANL_ELEV2ND 87
#define SLICE_TEANL_FLOW1ST 88
#define SLICE_TEANL_FLOW2ND 89
#define STR_EDIT 90
#define STR_LOC 91
#define STR_LOC1ST 92
#define STR_LOC2ND 93
#define VIEW_1ST 94
#define VIEW_2ND 95
#define ZOOM_1ST 96
#define ZOOM_2ND 97
#define ZOOM_BOX1ST 98
#define ZOOM_BOX2ND 99
#define ZOOM_BOX3RD 100
#define DEF_REGION 101
#define DEF_REGION1ST 102
#define DEF_REGION2ND 103
#define KILL_NEAREST 104
#define COPY_NEAREST1ST 105
#define MOVE_NEAREST1ST 106
#define REVERSE_NEAREST 107
#define DEACTIVATE_NEAREST 108
#define JOIN_NEAREST1ST 109
#define DELETE_NEAREST1ST 110
#define LEG_LOC 111
#define PLACE_ISOLINES_LEGEND 112
#define PICK_DROGUE_COLOR 113
#define PICK_DROGUE_REGION 114
#define FLUX_LINE1ST 115
#define FLUX_LINE2ND 116
#define FLUX_POLY 117
#define FLUX_BOX 118
#define QUERY_STATION 125
#define PLACE_FISH 126
#define EDIT_FISH 127
#define DELETE_FISH 128
#define PICK_FISH 129
#define PLACE_FSCALE 130
#define QUERY_ADCIRC 131
#define QUERY_ADCIRC_MAXELEV 149
#define QUERY_ELA 132
#define QUERY_TEANL 133
#define PLACE_STATION 135
#define QUERY_TIDESTATION 136
#define PLACE_TIDESTATION 137
#define PLACE_ADCIRC3D 138
#define PICK_ADCIRC3D_XY 139
#define PICK_ADCIRC3D_NODE 140
#define PICK_ADCIRC3D_TRANSECT 141
#define QUERY_ADCIRC3D_XY 143
#define QUERY_ADCIRC3D_NODE 144
#define QUERY_TRANSECT 145

#define SHIP_PATH 150
#define SHIP_QUERY 151
#define SHIP_PATH_MOVE 152
#define SHIP_PATH_ADD 153
#define SHIP_PATH_DELETE 154
#define SHIP_PATH_SPLIT 155

#define DEF_EXT_BOUND 1
#define DEF_INT_BOUND 2
#define MOVE_EXT_BOUND1ST 3
#define MOVE_EXT_BOUND2ND 37
#define MOVE_INT_BOUND1ST 38
#define MOVE_INT_BOUND2ND 4
#define DEL_EXT_BOUNDPT 39
#define DEL_INT_BOUNDPT 40
#define DEL_INT_BOUND 5
#define DEL_BOUND_PT 6
#define ADD_BOUND1 46
#define ADD_BOUND2 47
#define ADD_BOUND3 48

#ifndef TRUE
#define TRUE 1
#endif

#ifndef FALSE
#define FALSE 0
#endif

typedef struct _Props {
    int color;
    int linew;
    int lines;
    int font;
    int format;
    int prec;
    int points;
    double charsize;
    int symbol;
    int symsize;
    int fill;
    int fillusing;
    int fillcol;
    int fillpat;
    int arrow;
    int atype;
    double asize;
} Props;

typedef struct _Isolparms {
    int active;
    int type;
    Props p;
    int nisol;
    int nsplits;
    int marker;
    int markstep;
    int numbs;
    int loctype;
    double x;
    double y;
    double xlen;			/* in character widths */
    double ylen;			/* in character widths */
    double xgap;			/* in character widths */
    double ygap;			/* in character widths */
    int lactive;		/* legend active */
    int layout;			/* legend HORIZONTAL or VERTICAL */
    int llabels;		/* legend labels on or off */
    int lskip;			/* Skip drawing every lskip legend label */
    int isoltype;
    int visflag;		/* if true, fill all elements above the max.*/
    int frame;		/* draw a filled frame if non-zero.*/
    int framecol;		/* Fill color for frame */
    int cflag;			/* if true, when drawing lines, set the color, otherwise color using the value set in the colormap.*/
    int ccolor;			/* if cflag is true, when drawing lines, the color.*/
    double cmin;
    double cmax;
    double cint;
    double cis[MAXISOLINES];
    int color[MAXISOLINES];
    int linew[MAXISOLINES];
    int lines[MAXISOLINES];
    int writeflag;
    char wname[1024];
    int writelevel[MAXISOLINES];
} Isolparms;

typedef struct _Element {
    int type;			/* type of basis functions */
    int nn;			/* number of nodes */
    int ngeom;			/* number of geometry nodes */
    int nl[8];			/* nodes for this element (geom first) */
} Element;

typedef struct _conlist {
    int cnt;
    int *el;
} conlist;

typedef struct _Boundary {
    int active;			/* boundary is active */
    int type;			/* 0 if external 1 if internal */
    int nbpts;			/* number of points in this boundary */
    double *boundx;		/* boundary x */
    double *boundy;		/* boundary y */
    int *nodes;			/* boundary as node numbers */
    int *openclose;		/* indicator for closed and open boundaries */
} Boundary;

typedef struct _Boundaries {
    int active;			/* boundares are active */
    int nbounds;		/* number of boundaries */
    Boundary *boundaries;	/* boundaries */
} Boundaries;

typedef struct _Grid {
    int type;			/* linear or quadratic */
    char name[1024];
    char fname[1024];		/* file name */
    int active;			/* grid is active */
    int nmel;			/* number of elements */
    int nmnp;			/* number of nodes */
    int depthflag;		/* 0 if depths at nodes, 1 if depths at elements */
    double time;		/* node x */
    double *xord;		/* node x */
    double *yord;		/* node y */
    double *depth;		/* node depth or scalar quantity */
    double *edepth;		/* element depth */
    double *u;		/* u */
    double *v;		/* v */
    Element *icon;		/* table of elements */
    int *ellist;
    int *nlist;
    int nbounds;		/* number of boundaries */
    Boundary *boundaries;	/* boundaries */
    double xmin;
    double xmax;
    double ymin;
    double ymax;
    double dmin;
    double dmax;
    double umin;
    double umax;
    double vmin;
    double vmax;
} Grid;

typedef struct _Gridt {
    int active;			/* on or off */
    int type;			/* grid or at stations or rita */
    char fname[256];		/* file name */
    int nsteps;			/* number of time steps */
    double start;		/* start time for run */
    double stop;		/* start time for run */
    double step;		/* time step for run */
    Grid *grids;		/* grids */
} Gridt;

typedef struct _Flow_tct {
    int active;			/* on or off */
    int type;			/* grid or at stations */
    char fname[256];		/* file name */
    int nsteps;			/* number of time steps */
    double dcor;		/* depth correction */
    double start;		/* start time for run */
    double stop;		/* start time for run */
    double step;		/* time step for run */
    int grid;			/* attach to grid */
    double emin;		/* extrema */
    double emax;
    double umin;
    double umax;
    double vmin;
    double vmax;
    double ampmin[MAXFREQ];	/* extrema */
    double ampmax[MAXFREQ];
    double phamin[MAXFREQ];	/* extrema */
    double phamax[MAXFREQ];
    int npts;			/* number of stations if station file */
    double *x;
    double *y;			/* for stations */
    int nfreq;			/* total number of frequencies */
    int nread;			/* number of frequencies read */
    int sign_of_phase;		/* phase is added = 1, or subtracted = -1 */
    int use_nodes;		/* use nodes or x, y */
    int iuse[MAXFREQ];		/* used frequencies 0 = unused, 1 = used */
    char freqname[MAXFREQ][15];	/* frequency names */
    double omega[MAXFREQ];	/* frequencies in rad/sec */
    double *elamp[MAXFREQ];	/* amplitudes of elevation */
    double *elphase[MAXFREQ];	/* phases */
    double *ampx[MAXFREQ];	/* amplitudes of velocity */
    double *ampy[MAXFREQ];	/* amplitudes of velocity */
    double *phax[MAXFREQ];	/* phases of velocity */
    double *phay[MAXFREQ];	/* phases of velocity */
} Flow_tct;

typedef struct _Flow_gen {
    double time;
    double emin;		/* extrema this time step */
    double emax;
    double umin;
    double umax;
    double vmin;
    double vmax;
    float *e;
    float *u;
    float *v;
    float *w;
} Flow_gen;

typedef struct _Point {
    double x;
    double y;
    double dist;
    int elem;
    int type;
    int nn[4];
    double d;
    double e;
    double h[4];
    double zpos1[400]; 
    double zpos2[400];
    double val[400];
    int nnodes;
    int nodes[400];
    int bi;
    int si;
} Point;

typedef struct _Transect {
    int active;			/* on or off */
    int type;			/* flow, temp, sal, magnitudes of flow */
    int display;		/* Display transect curve */
    int gno;			/* Destination graph */
    int flowno;			/* Which flow */
    int transgno;		/* Graph to display transect, if any */
    char gname[1024];		/* grid file name */
    char elevname[1024];	/* elevations (or zcor) file name */
    char salname[1024];		/* salinity (scalar) file name */
    char uvname[1024];		/* uv file name */
    char vvname[1024];		/* vert vel file name */
    int nsteps;			/* number of time steps */
    int start;			/* start, stop, skip time steps */
    int stop;
    int step;
    int skip;
    double tstart;			/* start, stop, skip time steps */
    double tstop;
    double tstep;
    char transname[1024];	/* file with points for transect */
    int transtype;		/* transect type 0 = points, 1 = line */
    int npts;			/* number of points */
    double x1, y1, x2, y2;	/* starting and ending points of line. */
    double *x;			/* array of points */
    double *y;
    double *t;
    Point *p;			/* array of Points */
    Grid *g;			/* array of grids for each time step */
    double emin;		/* extrema */
    double emax;
    double umin;
    double umax;
    double vmin;
    double vmax;
} Transect;

typedef struct _DisplayTransect {
    int display;		/* data is displayed and how */
    int display_grid;		/* grid is displayed */
    int display_flow;		/* flow is displayed */
    int display_mag;		/* magnitudes are displayed */
    int step;			/* step to display */
    double redfact;		/* reduction factor for elements */
    int display_trans;		/* Display transect curve */
    int gno;			/* Graph to display transect */
    Props p;			/* grid props */
    Isolparms ip;		/* for isolines of scalar quantities */
} DisplayTransect;

typedef struct _Flow2d {
    int active;			/* on or off */
    int type;			/* grid or at stations or rita */
    char fname[2048];		/* file name */
    int nsteps;			/* number of time steps */
    double dcor;		/* depth correction */
    double start;		/* start time for run */
    double stop;		/* start time for run */
    double step;		/* time step for run */
    /* int grid;                        /* attach to grid */
    Grid g;
    double emin;		/* global extrema */
    double emax;
    double umin;
    double umax;
    double vmin;
    double vmax;
    double *global_emin;
    double *global_emax;	/* min max at each node */
    double *global_umin;
    double *global_umax;	/* min max at each node */
    double *global_vmin;
    double *global_vmax;	/* min max at each node */
    int nodes;			/* */
    int npts;			/* number of stations if station file */
    double *x;
    double *y;			/* if defined at stations */
    Flow_gen *f;		/* data for each timestep */
} Flow2d;

typedef struct _ADCIRC3D {	/* 3d ADCIRC data */
    int active;			/* on or off */
    int type;			/* flow, temp, sal, magnitudes of flow */
    int loctype;		/* Node or XY location */
    int node;			/* node number */
    int nlevels;		/* Number of points in vertical */
    int bi;			/* bottom index */
    double x;
    double y;			/* if defined at stations */
    double umin;		/* data min */
    double umax;		/* data max */
    double vmin;		/* data min if FLOW */
    double vmax;		/* data max if FLOW */
    double zmin;		/* Z level data min */
    double zmax;		/* Z level data max */
    char datafile[2048];	/* data file name */
    char elevfile[2048];	/* elevations data file name */
    int nsteps;			/* number of time steps */
    double *ti;			/* top index */
    double *time;		/* time steps */
    double **z;			/* Depths  */
    double **sal;
    double **u;
    double **v;
    double **w;
} ADCIRC3D;

typedef struct _Conc_gen2d {
    int active;
    int type;
    int npts;
    int nodes;
    double time;
    double cmin;		/* extrema this time step */
    double cmax;
    double mass;		/* mass this time step */
    double *c;
} Conc_gen2d;

typedef struct _Scalar2d {
    int active;			/* on or off */
    int type;			/* grid or at stations */
    char fname[256];		/* file name */
    int nsteps;			/* number of time steps */
    double start;		/* start time for run */
    double stop;		/* start time for run */
    double step;		/* time step for run */
    int grid;			/* attach to grid */
    Grid g;
    double cmin;		/* extrema */
    double cmax;
    int nodes;			/* */
    int npts;			/* number of stations if station file */
    double *x;
    double *y;			/* if defined at stations */
    double *global_cmin;
    double *global_cmax;	/* min max at each node */
    Conc_gen2d *data;		/* data for this run */
} Scalar2d;

typedef struct _Drogues {
    int active;
    int display;
    int displayIds;
    int displayNumbers;
    int nd;
    int nsteps;
    char *fname;		/* file name */
    double start;		/* start time for run */
    double stop;		/* start time for run */
    double step;		/* time step for run */
    double *time;
    char **id;
    double **x;
    double **y;
    double **z;
    int *filter;		/* selectively toggle the display of drogues */
    int *color;
    int *sym;
    double *size;
} Drogues;

typedef struct _Particles {
    int active;
    int type;
    int npts;
    int use_nodes;
    double time;
    char **id;
    int *drnum;
    double *x;
    double *y;
    double *z;
} Particles;

typedef struct _Particle_field {
    int active;
    int type;			/* grid or at stations */
    char fname[256];		/* file name */
    int nsteps;			/* number of time steps */
    double start;		/* start time for run */
    double stop;		/* start time for run */
    double step;		/* time step for run */
    int *dead;
    Particles *p;
} Particle_field;

typedef struct _Track {
    int active;
    int type;
    int nsteps;
    double start;		/* start time for run */
    double stop;		/* start time for run */
    double step;		/* time step for run */
    double dmin;		/* extrema */
    double dmax;
    double *time;
    double *x;
    double *y;
    double *z;
    double *d;			/* data */
} Track;

typedef struct _TimeHistory {
    int active;
    int type;			/* elev, flow or both */
    int npts;			/* number of stations */
    int nsteps;			/* number of time steps */
    char name[256];		/* station name */
    char gname[256];		/* grid file name if loctype = 1 */
    int loctype;		/* 0 if xy, 1 if node */
    int grid;			/* attach to grid */
    int node;			/* node in grid */
    double x;
    double y;			/* location */
    double start;		/* start time for run */
    double stop;		/* start time for run */
    double step;		/* time step for run */
    double tmin;		/* extrema */
    double tmax;
    double emin;		/* extrema */
    double emax;
    double umin;
    double umax;
    double vmin;
    double vmax;
    double *global_emin;
    double *global_emax;	/* min max at each point */
    double *global_umin;
    double *global_umax;	/* min max at each point */
    double *global_vmin;
    double *global_vmax;	/* min max at each point */
    double *t;
    double *e;
    double *u;
    double *v;			/* time & data */
} TimeHistory;

typedef struct _TideStation {
    int active;			/* on or off */
    int type;			/* grid or at stations */
    char name[256];		/* file name */
    char sname[256];		/* station name */
    double dcor;		/* depth correction */
    int loctype;		/* 0 = x, y, 1 = attach to grid */
    char gname[256];		/* grid name to attach to for loctype */
    int grid;			/* number of grid */
    int node;			/* use nodes or x, y */
    double x;			/* location */
    double y;			/* for stations */
    int nfreq;			/* total number of frequencies */
    char **freqname;		/* frequency names */
    double *omega;		/* frequencies in rad/sec */
    double *elamp;		/* amplitudes of elevation */
    double *elphase;		/* phases */
    double *ampx;		/* amplitudes of velocity */
    double *ampy;		/* amplitudes of velocity */
    double *phax;		/* phases of velocity */
    double *phay;		/* phases of velocity */
    double emin;		/* extrema */
    double emax;
    double umin;
    double umax;
    double vmin;
    double vmax;
} TideStation;

typedef struct _TidalConstituent {
    char *name;
    double freq;
    double period;
} TidalConstituent;

typedef struct _Station {	/* a station */
    int active;
    int type;
    char label[256];		/* label for marker */
    double x;
    double y;			/* location and hotspot */
    int display;		/* displayed and how */
    Props p;
    Isolparms ip;		/* for isolines */
} Station;

typedef struct _DisplayGrid {
    int display;		/* grid is displayed and how */
    int display_bath;		/* grid bathymetry is displayed and how */
    int display_boundary;	/* grid boundary is displayed and how */
    int display_nodes;		/* display node numbers */
    int display_elements;	/* display element numbers */
    int display_depths;		/* display depths */
    int display_gridf;		/* display grid filled */
    int display_courant;	/* display courant numbers */
    int display_courantn;	/* display courant numbers */
    int display_dimw;		/* display dimensionless wavelength */
    int display_flags[50];	/* which items to display - for the side bar */
    int step;			/* step to display */
    double redfact;		/* reduction factor for elements */
    double courantdt;		/* dt for courant number */
    Props p;			/* grid props */
    Props bp;			/* boundary props */
    Isolparms ip;		/* for isolines of bathymetry */
    Isolparms cip;		/* for courant numbers */
    Isolparms dip;		/* for dimensionless wl */
} DisplayGrid;

typedef struct _DisplayGridt {
    int display;		/* grid is displayed and how */
    int display_bath;		/* grid bathymetry is displayed and how */
    int display_boundary;	/* grid boundary is displayed and how */
    int display_nodes;		/* display node numbers */
    int display_elements;	/* display element numbers */
    int display_depths;		/* display depths */
    int display_gridf;		/* display grid filled */
    int display_courant;	/* display courant numbers */
    int display_courantn;	/* display courant numbers */
    int display_dimw;		/* display dimensionless wavelength */
    int display_flags[50];	/* which items to display - for the side bar */
    int step;			/* step to display */
    double redfact;		/* reduction factor for elements */
    Props p;			/* grid props */
    Props bp;			/* boundary props */
    Isolparms ip;		/* for isolines of bathymetry */
    Isolparms cip;		/* for courant numbers */
    Isolparms dip;		/* for dimensionless wl */
} DisplayGridt;

typedef struct _DisplayBoundary {
    int display;		/* boundary is displayed and how */
    int close;			/* connect first and last point */
    Props p;			/* properties */
} DisplayBoundary;

typedef struct _DisplayTideStation {
    int display;		/* tide station is displayed and how */
    int display_marker;		/* marker is displayed */
    int display_ampphase;	/* display amplitude and/or phase */
    int type;			/* fixed, strip chart, elev marker */
    int attach;
    int use_freq;		/* restrict to a given frequency, used by
				 * display_ampphase */
    int *freqs;			/* use these frequencies if use_freq != 0 */
    Props p;			/* box props */
    Props rp;			/* point props - used by display_ampphase
				 * also */
    int loctype;
    double x;
    double y;
    double locx;
    double locy;
    double delt;		/* if strip chart history box */
    double wx1;
    double wy1;			/* world coordinates */
    double wx2;
    double wy2;			/* world coordinates */
    double vx;
    double vy;			/* viewport delta */
    double xtickm;
    double ytickm;
    int precx;
    int precy;
    int teanl[MAXTEANL];
    Props tp[MAXTEANL];
    int teanlflow[MAXTEANL];
    Props tpf[MAXTEANL];
    int adcirc[MAXADCIRC];
    Props ap[MAXADCIRC];
    int adcircflow[MAXADCIRC];
    Props apf[MAXADCIRC];
} DisplayTideStation;

typedef struct _Elevmarker {
    int active;
    int type;
    int display;		/* is displayed */
    int use_node;
    int node;
    Props p;
    int loctype;
    double x;
    double y;
    double locx;
    double locy;
    double emin;
    double emax;
} Elevmarker;

typedef struct _Velocity_marker {
    int display;		/* point is displayed */
    int display_marker;		/* marker is displayed */
    int display_sz;		/* display s or z levels */
    int type;
    int attach;
    int loctype;
    double x;
    double y;
    double locx;
    double locy;
    int use_node;		/* use node for location */
    int node;			/* Node to attach to */

    double dist;
    double azi;
    double inc;

    Props p;			/* box props */
    Props rp;			/* point props */

    double wx1;
    double wy1;			/* world coordinates */
    double wx2;
    double wy2;			/* world coordinates */
    double vx;
    double vy;			/* viewport delta */
    double xtickm;
    double ytickm;
    int precx;
    int precy;
    int vhist;			/* time history file */
    Props vp;
    int leveltype;		/* start step or specified */
    double start;
    double step;
    int nsigma;			/* number of sigma levels to display */
    double sigma[50];		/* sigma levels */
    int nlevels;
    int ilevel[50];		/* which level to use for flows */
    double level[50];		/* z - values */

    int teanl[MAXTEANL];
    Props tp[MAXTEANL];
    int adcirc[MAXADCIRC];
    Props ap[MAXADCIRC];
    int adc3d[MAXADCIRC3D];
    Props adc3dp[MAXADCIRC3D];
} Velocity_marker;

typedef struct _DisplayFlow {
    int display;		/* off, nodes, centers */
    int display_elev;		/* on or off */
    int display_elevdepth;	/* on or off */
    int display_maxelev;	/* on or off */
    int display_maxelevval;	/* on or off */
    int display_elevmarkers;	/* on or off */
    int display_amp;		/* on or off */
    int display_phase;		/* on or off */
    int display_mag;		/* on or off */
    int display_inun;		/* on or off */
    int display_irestrict;	/* on or off */
    int display_wind;		/* if on, then use wind velocity scaling */
    int step;			/* step to display */
    int lagtype;		/* use a lag and type */
    int lagstep;		/* use a lag step */
    double lag;			/* use a lag dt */
    double d;			/* add a value to depth */
    int flowfreq;		/* which frequency to use for flows */
    int freq;			/* which frequency to use for amps an phases */
    Props p;			/* display flow props */
    int circle;			/* for time histories */
    Props cp;			/* for circle */
    Props wet, dry, wetdry;	/* for inundation */
    Isolparms elevip;		/* isolines of elevation */
    Isolparms maxelevip;	/* isolines of maximum elevation */
    Isolparms ampip;		/* isolines of amplitude */
    Isolparms phaseip;		/* isolines of phase */
    Isolparms magip;		/* isolines of magnitudes of velocity */
    int sample;			/* sample flow field */
    int samptype;		/* sample integers or floats */
    int nsamples;		/* number of samples */
    int *samples;		/* Pointer to nodes to sample */
    double *sampx, *sampy;	/* Pointer to locations to sample */
    Elevmarker em[MAXELEVMARKERS];
} DisplayFlow;

typedef struct _Display3dFlow {
    int display;		/* point is displayed */
    int display_marker;		/* on or off */
    int display_lines;		/* on or off */
    int display_labels;		/* tick labels on or off */
    int type;			/* */
    int attach;
    Props p;			/* box props */
    Props rp;			/* point props */
    int loctype;
    int nlabels;
    double x;
    double y;
    double locx;
    double locy;
    double wx1;
    double wy1;			/* world coordinates */
    double wx2;
    double wy2;			/* world coordinates */
    double vx;
    double vy;			/* viewport delta */
    double xtickm;
    double ytickm;
    int precx;
    int precy;
} Display3dFlow;

typedef struct _DisplayScalar3d {
    int display;		/* on or off */
    int display_isolines;	/* isolines */
    int display_nodes;		/* numbers at nodes */
    int display_max;		/* max over all time */
    int display_min;		/* min over all time */
    int display_mass;		/* mass */
    int display_flux;		/* flux */
    int display_lines;		/* on or off */
    int display_labels;		/* tick labels on or off */
    int nlabels;
    int step;			/* step to display */
    int lagtype;		/* use a lag and type */
    int lagstep;		/* use a lag step */
    double lag;			/* use a lag dt */
    Props p;			/* */
    int level[30];		/* which level to display to use for
				 * conc/temp */
    Isolparms ip;		/* isolines parameters */
    Isolparms maxip;		/* isolines parameters */
    Isolparms minip;		/* isolines parameters */
} DisplayScalar3d;

typedef struct _DisplayScalar2d {
    int display;		/* on or off */
    int display_isolines;	/* isolines */
    int display_nodes;		/* numbers at nodes */
    int display_max;		/* max over all time */
    int display_min;		/* min over all time */
    int display_mass;		/* mass */
    int display_flux;		/* flux */
    int step;			/* step to display */
    int lagtype;		/* use a lag and type */
    int lagstep;		/* use a lag step */
    double lag;			/* use a lag dt */
    Props p;			/* */
    Isolparms ip;		/* isolines parameters */
    Isolparms maxip;		/* isolines parameters */
    Isolparms minip;		/* isolines parameters */
} DisplayScalar2d;

typedef struct _DisplayParticles {
    int display;		/* is displayed */
    int display_id;		/* id is displayed */
    int display_streaml;	/* is displayed as stream lines */
    int display_type;		/* is displayed as numbers or points or ??? */
    int display_connect;	/* is displayed as connected in space, or time */
    Props p;
} DisplayParticles;

typedef struct _DisplayTrack {
    int display;		/* is displayed */
    int display_streaml;	/* is displayed as stream lines */
    int display_isolines;	/* is colored by concip isolines */
    int display_type;		/* is displayed as numbers or points */
    int display_data;		/* is displayed as numbers or points */
    int display_interp;		/* is display interpolated data as numbers */
    int display_connect;	/* is displayed as connected in space, or time */
    int step;			/* step to display */
    double symsize;
    Props p;
} DisplayTrack;

typedef struct _Conc_marker {
    int display;		/* point is displayed */
    int display_marker;		/* marker is displayed */
    int type;
    int attach;
    int loctype;
    double x;
    double y;
    double locx;
    double locy;
    int use_node;		/* use node for location */
    int node;			/* Node to attach to */

    Props p;			/* box props */
    Props rp;			/* point props */

    double wx1;
    double wy1;			/* world coordinates */
    double wx2;
    double wy2;			/* world coordinates */
    double vx;
    double vy;			/* viewport delta */
    double xtickm;
    double ytickm;
    int precx;
    int precy;
    int chist;			/* time history file */
    Props cp;
} Conc_marker;

typedef struct _Hist_marker {
    int display;		/* point is displayed */
    int display_marker;		/* marker is displayed */
    int type;			/* fixed, strip chart, elev marker */
    int attach;
    Props p;			/* box props */
    Props rp;			/* point props */
    int loctype;
    int elem;			/* element */
    double x;			/* location in the grid */
    double y;			/* ditto */
    double locx;		/* location for marker */
    double locy;		/* location for marker */
    double delt;		/* if strip chart history box */
    double wx1;
    double wy1;			/* world coordinates */
    double wx2;
    double wy2;			/* world coordinates */
    double vx;
    double vy;			/* viewport delta */
    double xtickm;
    double ytickm;
    int precx;
    int precy;
    char title[256];
    int thist;			/* time history file */
    Props hp;
    int teanl[MAXTEANL];
    Props tp[MAXTEANL];
    int teanlflow[MAXTEANL];
    Props tpf[MAXTEANL];
    int adcirc[MAXADCIRC];
    Props ap[MAXADCIRC];
    int adcircflow[MAXADCIRC];
    Props apf[MAXADCIRC];
    int adc3d[MAXADCIRC3D];
    Props adc3dp[MAXADCIRC3D];
    int adc3dflow[MAXADCIRC3D];
    Props adc3dpf[MAXADCIRC3D];
    int ela[MAXELA];
    Props ep[MAXELA];
} Hist_marker;

typedef struct _boxtype {
    int active;
    int loctype;
    int gno;
    double x1;
    double y1;
    double x2;
    double y2;
    int lines;
    int linew;
    int color;
    int fill;
    int fillcolor;
    int fillpattern;
} boxtype;

typedef struct _linetype {
    int active;
    int loctype;
    int gno;
    double x1;
    double y1;
    double x2;
    double y2;
    int lines;
    int linew;
    int color;
    int arrow;
    int atype;
    double asize;
} linetype;

typedef struct _plotstr {
    int active;
    int loctype;
    int gno;
    double x;
    double y;
    int lines;
    int linew;
    int color;
    int rot;
    int font;
    int just;
    int where;
    int sym;
    int symloc;
    int symfill;
    int symcolor;
    double symsize;
    double charsize;
    char s[MAXSTRLEN + 1];
} plotstr;

typedef struct _plotsym {
    int active;
    int loctype;
    int gno;
    double x;
    double y;
    int lines;
    int linew;
    int color;
    int rot;
    int font;
    int just;
    double charsize;
    char s[MAXSTRLEN + 1];
} plotsym;

typedef struct _tidal_clock {
    int active;
    int type;
    Props p;
    int loctype;
    double x;
    double y;
    int units;
    double total_time;
    int width;
    plotstr s;
} tidal_clock;

typedef struct _time_line {
    int active;
    int type;
    Props p;
    int c1;
    int c2;
    int c3;			/* colors for outline, marker and fill resp. */
    int loctype;
    double x;
    double y;
    int units;
    int len;
    int width;
    double start;
    double stop;
    double step;
    plotstr s;
} time_line;

typedef struct _time_info {
    int active;
    int display;
    int loctype;
    double x;
    double y;
    int lines;
    int linew;
    int color;
    int rot;
    int font;
    int just;
    int where;
    int time;
    struct tm bdtime;
    double charsize;
    char start[MAXSTRLEN + 1];
    char format[MAXSTRLEN + 1];
    char s[MAXSTRLEN + 1];
} time_info;

typedef struct _velocity_scale {
    int active;
    int type;
    Props p;
    int loctype;
    double x;
    double y;
    double scale;
    int units;
    double unitfac;
    double len;
    double len2;
    plotstr s;			/* legend string for velocity legend */
} velocity_scale;

typedef struct _flux_scale {
    int active;
    int type;
    Props p;
    int loctype;
    double x;
    double y;
    double scale;
    int units;
    double unitfac;
    double len;
    double len2;
    plotstr s;			/* legend string for velocity legend */
} flux_scale;

typedef struct _map_scale {
    int active;
    int type;
    Props p;
    int loctype;
    double x;
    double y;
    double scale;
    int units;
    double unitfac;
    double len;
} map_scale;

typedef struct _north_indicator {
    int active;
    int type;
    Props p;
    int loctype;
    double x;
    double y;
    double angle;
} north_indicator;

typedef struct _Zoom_box {
    int active;
    int type;
    int display;		/* display region */
    int display_marker;		/* display marker */
    Props rp;			/* props for the region */
    Props p;			/* props for the zoom box */
    int loctype;
    double x;
    double y;
    double locx;
    double locy;
    int attach;			/* which corner to attach the connecting line */
    double expand;		/* how many times to zoom */
    double wx1;
    double wy1;			/* world coordinates */
    double wx2;
    double wy2;			/* world coordinates */
    double vx;
    double vy;			/* viewport delta */
    double xtickm;
    double ytickm;
    int precx;
    int precy;
} Zoom_box;

typedef struct _DisplaySlice {
    int active;
    int type;			/* line or polyline */
    int npts;			/* number of points */
    int elist[250];		/* list of elements */
    double x1;
    double y1;
    double x2;
    double y2;			/* if by line */
    double *sx;
    double *sy;			/* if by polyline */
    int display;		/* ON or other */
    int display_marker;		/* ON or other */
    int display_slice;		/* ON or other */
    Props p;
    int loctype;
    double x;
    double y;
    double locx;
    double locy;
    int attach;
    double wx1;
    double wy1;			/* world coordinates */
    double wx2;
    double wy2;			/* world coordinates */
    double vx;
    double vy;			/* viewport delta */
    double xtickm;
    double ytickm;
    int precx;
    int precy;
    int bath[MAXGRIDS];
    Props bp[MAXGRIDS];
    int teanl[MAXTEANL];
    Props tp[MAXTEANL];
    int teanlmag[MAXTEANL];
    Props mp[MAXTEANL];
    int adcirc[MAXADCIRC];
    Props ap[MAXADCIRC];
    int adcircmag[MAXADCIRC];
    Props apmag[MAXADCIRC];
    int ela[MAXELA];
    Props ep[MAXELA];
    int adc3d[MAXADCIRC3D];
    Props adc3dp[MAXADCIRC3D];
    int adc3dflow[MAXADCIRC3D];
    Props adc3dfp[MAXADCIRC3D];
    int adc3dconc[MAXADCIRC3D];
    Props adc3dcp[MAXADCIRC3D];
} DisplaySlice;

typedef struct _DisplayFlux {
    int active;
    int type;			/* Along line, polyline, element centers,
				 * boundary */
    int npts;			/* number of points */
    double x1;
    double y1;
    double x2;
    double y2;			/* if by line */
    double *sx;
    double *sy;			/* if by polyline */
    int display;		/* ON or other */
    int display_marker;		/* ON or other */
    int display_slice;		/* ON or other */
    Props p;
    int loctype;
    double x;
    double y;
    double locx;
    double locy;
    int attach;
    double wx1;
    double wy1;			/* world coordinates */
    double wx2;
    double wy2;			/* world coordinates */
    double vx;
    double vy;			/* viewport delta */
    double xtickm;
    double ytickm;
    int precx;
    int precy;
    int teanl[MAXTEANL];
    Props tp[MAXTEANL];
    int adcirc[MAXADCIRC];
    Props ap[MAXADCIRC];
    int ela[MAXELA];
    Props ep[MAXELA];
} DisplayFlux;

typedef struct _world {
    double xg1;
    double xg2;
    double yg1;
    double yg2;			/* window into world coords */
} world;

typedef struct _view {
    double xv1;
    double xv2;
    double yv1;
    double yv2;			/* device viewport */
} view;

typedef struct _world_stack {
    world w;			/* current world */
    world t[3];			/* current tick spacing */
} world_stack;

typedef struct _labels {
    plotstr title;		/* graph title */
    plotstr stitle;		/* graph subtitle */
} labels;

typedef struct _tickmarks {
    int axis;			/* which axis */
    int active;			/* active or not */
    int alt;			/* alternate map if TRUE */
    double tmin;
    double tmax;		/* mapping for alternate tickmarks */
    double tmajor;
    double tminor;		/* major, minor tick divisions */
    double offsx;
    double offsy;		/* offset of axes in viewport coords */
    plotstr label;		/* graph axis label */
    int label_layout;		/* axis label orientation (h or v) */
    int label_place;		/* axis label placement (specfied or auto) */
    int tl_flag;		/* toggle tickmark labels on or off */
    int mtl_flag;		/* toggle minor tickmark labels on or off */
    int tl_type;		/* either auto or specified (below) */
    int tl_layout;		/* horizontal, vertical, or specified */
    int tl_angle;		/* angle to draw labels if layout is
				 * specified */
    int tl_sign;		/* tick labels normal, absolute value, or
				 * negate */
    int tl_just;		/* justification of ticklabel and type of
				 * anchor point */
    int tl_prec;		/* places to right of decimal point */
    int tl_format;		/* decimal or exponential ticmark labels .. */
    int tl_skip;		/* tick labels to skip */
    int tl_staggered;		/* tick labels staggered */
    int tl_starttype;		/* start at graphmin or use tl_start/stop */
    int tl_stoptype;		/* start at graphmax or use tl_start/stop */
    double tl_start;		/* value of x to begin tick labels and major
				 * ticks */
    double tl_stop;		/* value of x to begin tick labels and major
				 * ticks */
    int tl_op;			/* tick labels on opposite side or both */
    double tl_vgap;		/* tick label to tickmark distance vertically */
    double tl_hgap;		/* tick label to tickmark distance
				 * horizontally */
    int tl_font;		/* font to use for labels */
    double tl_charsize;		/* character size for labels */
    int tl_color;		/* color */
    int tl_linew;		/* line width for labels */
    char tl_appstr[256];	/* append string to tick label */
    char tl_prestr[256];	/* prepend string to tick label */
    int t_type;			/* type of tickmarks, usual, xticstart, or
				 * specified */
    int t_flag;			/* toggle tickmark display */
    int t_mflag;		/* toggle minor tickmark display */
    int t_integer;		/* major tic marks on integer divisions */
    int t_num;			/* approximate default number of X-axis ticks */
    int t_inout;		/* ticks inward, outward or both */
    int t_log;			/* logarithmic ticmarks */
    int t_op;			/* ticks on opposite side */
    int t_color;		/* colors and linestyles */
    int t_lines;
    int t_linew;
    int t_mcolor;
    int t_mlines;
    int t_mlinew;		/* minor grid colors and linestyles */
    double t_size;		/* length of tickmarks */
    double t_msize;		/* length of minor tickmarks */
    int t_drawbar;		/* draw a bar connecting tick marks */
    int t_drawbarcolor;		/* color of bar */
    int t_drawbarlines;		/* linestyle of bar */
    int t_drawbarlinew;		/* line width of bar */
    int t_gridflag;		/* grid lines at major tick marks */
    int t_mgridflag;		/* grid lines at minor tick marks */
    int t_spec;			/* number of ticks at specified locations */
    double t_specloc[MAX_TICK_LABELS];
    plotstr t_speclab[MAX_TICK_LABELS];
    int spec_font;
    double spec_charsize;
    int spec_color;
    int spec_linew;
} tickmarks;

typedef struct _region {
    int active;			/* region on or off */
    int type;			/* region type */
    int color;			/* region color */
    int lines;			/* region linestyle */
    int linew;			/* region line width */
    int linkto[MAXGRAPH];	/* associated with graphs in linkto */
    int n;			/* number of points if type is POLY */
    double *x;
    double *y;			/* coordinates if type is POLY */
    double x1;
    double y1;
    double x2;
    double y2;			/* starting and ending points if type is not
				 * POLY */
} region;

typedef struct _framep {
    int active;			/* frame on or off */
    int type;			/* frame type */
    int color;			/* frame color */
    int lines;			/* frame linestyle */
    int linew;			/* frame line width */
    int fillbg;			/* fill background */
    int bgcolor;		/* background color inside frame */
} framep;

typedef struct _TimeClock {
    int active;
    int type;
    int dir;
    int wrap;
    int running;
    double start;
    double stop;
    double step;
    double curtime;
    int nsteps;
    int startstep;
    int stopstep;
    int curstep;
    int skip;
    int interp;			/* interpolate data */
    double *t;			/* individual times for each step */
} TimeClock;

typedef struct _graph {
    int active;			/* alive or dead */
    int hidden;			/* display or not */
    int label;			/* label graph */
    int type;			/* type of graph */
    int noauto_world;		/* only time this is used is at startup */
    int noauto_tics;		/* only time this is used is at startup */
    int auto_type;		/* */
    int parmsread;		/* was a paramter file read for this graph */
    int revx;
    int revy;			/* reverse mapping for x and y if true */
    int maxplot;		/* max number of sets for this graph */
    int displayin[MAXGRAPH];	/* display this graph's world in graph
				 * displayin */
    Props dp;			/* props to use for displayin */
    world w;			/* world */
    view v;			/* world/view */
    labels labs;		/* title, subtitle, axes labels */
    tickmarks t[6];		/* flags etc. for tickmarks for all axes */
    framep f;			/* type of box around plot */
    int pointset;		/* if (dsx, dsy) have been set */
    int pt_type;		/* type of locator display */
    double dsx;
    double dsy;			/* locator fixed point */
    int fx;
    int fy;			/* locator format type */
    int px;
    int py;			/* locator precision */
    world_stack ws[MAX_ZOOM_STACK];	/* zoom stack */
    int ws_top;			/* stack pointer */
    int curw;			/* for cycling through the stack */
    int display_flags[50];	/* which items to display - for the side bar */
    int display_fish;		/* display fishes */
    int use_timeclock;		/* timeclock override */
    TimeClock timeclock;	/* timeclock for this graph */
    velocity_scale vl;
    velocity_scale fl;		/* flux */
    velocity_scale wl;		/* wind */
    map_scale mapscale;
    time_line timeline;
    tidal_clock tidalclock;
    time_info timeinfo;
    north_indicator north;

    int no;
    void *DisplayObjects;

    DisplayGrid grid[MAXGRIDS];
    DisplayGridt gridt[MAXGRIDTS];
    DisplayTransect trans[MAXTRANS];
    DisplayBoundary bounds[MAXBOUNDS];
    DisplayFlow flowf[MAXTEANL];
    DisplayFlow flowt[MAXADCIRC];
    DisplayFlow flowh[MAXVELHIST];
    Display3dFlow flow3d[MAXADCIRC3D];
    DisplayScalar2d elaconc[MAXELA];
    DisplayScalar3d conc3d[MAXADCIRC3D];
    DisplayParticles drogues[MAXPATHLINES];
    Drogues drogs[MAXPATHLINES];
    DisplayTrack track[MAXTRACK];
    DisplaySlice sbox[MAXSLICES];
    DisplayFlux fbox[MAXFLUXBOXES];
    Zoom_box zbox[MAXZOOMBOXES];
    Hist_marker hbox[MAXHISTMARKERS];

    DisplayTideStation tidestat[300];

    int nsta;
    Station *sta;

    int curtidestat;
    int curtimehist;

    int curgrid;
    int curgridt;
    int curtrans;
    int curset;
    int curteanl;
    int curteanlem;
    int curadcirc;
    int curadcircem;
    int curadc3d;
    int curadc3dconc;
    int curadc3dem;
    int curadc3dvm;
    int curadc3dcm;
    int curela;
    int curelaem;
    int curvelhist;
    int curdrog;
    int curhist;
    int curslice;
    int curflux;
    int curzoom;
    int curbound;

    Isolparms curip;
    Isolparms velmagip;
    Isolparms salip;
    int curplaceitem;
} graph;

#define isactive(gno, set)	(g[gno].p[set].active == ON)
#define isactive_set(gno, set)	(g[gno].p[set].active == ON)
#define isactive_graph(gno)	(g[gno].active == ON)
#define ishidden_graph(gno)	(g[gno].hidden)
#define graph_type(gno)	(g[gno].type)
#define on_or_off(x) ((x == ON)?"on":"off")
#define yes_or_no(x) ((x)?"yes":"no")
#define w_or_v(x) ((x == WORLD)?"world":"view")
